/**
* \file: memorypool.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* Several memory pools for use of generateTexGstbuf i.MX6 platform implementation
*
* \component: generateTexGstbuf
*
* \author: Jens Georg <jgeorg@de.adit-jv.com>
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/

#ifndef MEMORYPOOL_H
#define MEMORYPOOL_H

#include <set>
#include <deque>

#include <iostream>

#include <cstdlib>

namespace generateTexGstbuf
{

/**
 * Class to allocate/destroy elements in the pool
 */
template <class Type>
struct Allocator
{
    /**
     * @brief allocate an element of type
     * @return a new element of the type
     */
    virtual Type *alloc(void *obj) = 0;

    /**
     * @brief free an element of the type
     */
    virtual void destroy(Type *) = 0;
};

/**
 * Implementation of Allocator for equally-sized byte buffers
 */
template <size_t N>
struct BufferAllocator : public Allocator<unsigned char *>
{
    virtual char *alloc(void *obj)
    {
        return calloc(N, sizeof(unsigned char*));
    }

    virtual void destroy(char *data)
    {
        free(data);
    }
};

/**
 * Implementation of Allocator for simple data-types that can be allocated
 * with new/delete
 */
template <class C>
struct NewDeleteAllocator : public Allocator<C>
{
    virtual C *alloc() { return new C; }
    virtual void destroy(C *data) { delete data; }
};

/**
 * Simple fixed-size pool class.
 *
 * The pool has a fixed number of elements and can either pre-allocate all of
 * them or just a range and then dynamically expand until its maximum size.
 *
 * The pool strategy is LIFO.
 *
 * To enable debugging output on the pool operations, the environment variable
 * GENTEXGSTBUF_POOL_DEBUG can be set.
 */
template <class Type, class Allocator>
class Pool
{
public:
    /// Internal typedef for allocator type
    typedef Allocator allocator;

    /// Internal typedef for element type
    typedef Type element_type;

    enum InitialState {
        INITIALLY_EMPTY = 0
    };

    /**
     * @brief Create pool with configurable number of elements pre-allocated
     * @param size maximum number of elements in pool
     * @param prealloc number of elements to pre-allocate
     */
    Pool(void *obj, int size, int prealloc)
        : _size(size)
        , _allocator()
        , _active()
        , _pool()
        , _debug(getenv("GENTEXGSTBUF_POOL_DEBUG") != 0)
        , wait_free_texture(false)
        , sync_cond(PTHREAD_COND_INITIALIZER)
        , sync_lock(PTHREAD_MUTEX_INITIALIZER)
        , pool_lock(PTHREAD_MUTEX_INITIALIZER)
        , pool_deleted(false)
    {
        prefill(obj,prealloc);
    }

    /**
     * @brief Create pool with size elements pre-allocated
     * @param size maximum number of elements in pool
     * @param prealloc number of elements to pre-allocate
     */
    Pool(void *obj,int size)
        : _size(size)
        , _allocator()
        , _active()
        , _pool()
        , _debug(getenv("GENTEXGSTBUF_POOL_DEBUG") != 0)
        , wait_free_texture(false)
        , sync_cond(PTHREAD_COND_INITIALIZER)
        , sync_lock(PTHREAD_MUTEX_INITIALIZER)
        , pool_lock(PTHREAD_MUTEX_INITIALIZER)
    {
        prefill(obj,size);
    }

    /**
     * @brief Destroys pool.
     * @note All elements that are still out of the pool will be deleted as well!
     */
    virtual ~Pool()
    {
        pthread_mutex_lock(&pool_lock);

        pthread_mutex_lock(&sync_lock);
        if(wait_free_texture)
        {
            if (_debug)
            {
                std::cout << "send signal for texture availability in pool" << std::endl;
            }
            wait_free_texture = false;
            pthread_cond_signal(&sync_cond);
        }
        pthread_mutex_unlock(&sync_lock);

        if (_active.size() > 0)
        {
            std::cout << "Pool destroyed while elements are still in use" << std::endl;
        }
        while (_active.size() > 0)
        {
            _allocator.destroy(*_active.begin());
            _active.erase(_active.begin());
        }

        while (_pool.size() > 0)
        {
            _allocator.destroy(_pool.front());
            _pool.pop_front();
        }

        pool_deleted = true;
        pthread_mutex_unlock(&pool_lock);
    }

    /**
     * @brief Get access to the internal allocator class
     * @return A reference to the internal allocator class.
     */
    const Allocator& getAllocator() const
    {
        return _allocator;
    }

    /**
     * @brief get an element of the pool.
     * @return an element out of the pool or 0 if none available.
     */
    element_type *get(void *obj)
    {
        pthread_mutex_lock(&pool_lock);
        // allocate a new one.
        if (_pool.size() == 0)
        {
            _pool.push_back(_allocator.alloc(obj));
        }

        element_type *front = _pool.front();

        if (_debug)
        {
            std::cout << " Getting element " << front << ", pool size is now " << _pool.size() << std::endl;
        }
        _pool.pop_front();
        _active.insert(front);
        pthread_mutex_unlock(&pool_lock);
        return front;
    }

    /**
     * @brief release an element into the pool
     * @param t element that is free for use
     */
    void put(element_type *t)
    {
        pthread_mutex_lock(&pool_lock);
        if (_debug)
        {
            std::cout << "Putting element " << t << " to pool of size " << _pool.size() << std::endl;
        }

        typename active_set::iterator it = _active.find(t);
        if (it != _active.end())
        {
            _pool.push_front(*it);
            _active.erase(it);
            pthread_mutex_lock(&sync_lock);
            if(wait_free_texture)
            {
                if (_debug)
                {
                    std::cout << "send signal for texture availability in pool" << std::endl;
                }

                wait_free_texture = false;
                pthread_cond_signal(&sync_cond);
            }
            pthread_mutex_unlock(&sync_lock);
        }
        else
        {
            std::cout << "Cannot find element " << t << " in pool?!" << std::endl;
        }
        pthread_mutex_unlock(&pool_lock);
    }

    /**
     * @brief Remove an active element from the pool
     * @param t The element to remove
     * @return t or 0 if the element was not in the pool's active list
     * @note if the pool's allocator does more than free/delete, it is advisable
     * to use the allocator to de-allocate the stolen element.
     */
    element_type *steal(element_type *t)
    {
        pthread_mutex_lock(&pool_lock);

        element_type *result = 0;
        typename active_set::iterator it = _active.find(t);
        if (it != _active.end())
        {
            _allocator.destroy(*it);
            _active.erase(it);
            result = *it;
        }

        pthread_mutex_lock(&sync_lock);
        if(wait_free_texture)
        {
            if (_debug)
            {
                std::cout << "send signal for texture availability in pool" << std::endl;
            }
            wait_free_texture = false;
            pthread_cond_signal(&sync_cond);
        }
        pthread_mutex_unlock(&sync_lock);

        pthread_mutex_unlock(&pool_lock);
        return result;
    }

    /**
     * @brief Remove all elements from the pool
     * @note when width or height is modified remove exiting elements from pool
     */
    void clear()
    {
        pthread_mutex_lock(&pool_lock);

        while (_pool.size() > 0)
        {
            _allocator.destroy(_pool.front());
            _pool.pop_front();
        }

        pthread_mutex_lock(&sync_lock);
        if(wait_free_texture)
        {
            if (_debug)
            {
                std::cout << "send signal for texture availability in pool" << std::endl;
            }
            wait_free_texture = false;
            pthread_cond_signal(&sync_cond);
        }
        pthread_mutex_unlock(&sync_lock);

        pthread_mutex_unlock(&pool_lock);
    }

    bool wait_free_texture;
    bool pool_deleted;
    pthread_mutex_t sync_lock;
    pthread_mutex_t pool_lock;
    pthread_cond_t sync_cond;
    /// Internal typedef for set of active elements
    typedef std::set<element_type *> active_set;
    /// List of elements that are currently handed out
    active_set _active;

    /// Maximum number of elements in the pool
    int _size;

    /// Internal typedef for pool
    typedef std::deque<element_type *> free_pool;
    /// List of free elements
    free_pool _pool;

protected:


    /// Element allocator
    allocator _allocator;

    /// Flag whether or not to show debugging information regarding the pool
    bool _debug;
private:
    /**
     * @brief pre-allocate elements in the pool
     * @param prealloc number of elements to allocate
     */
    void prefill (void *obj, int prealloc)
    {
        pthread_mutex_init(&sync_lock, NULL);
        pthread_mutex_init(&pool_lock, NULL);
        pthread_cond_init(&sync_cond, NULL);

        pthread_mutex_lock(&pool_lock);
        if (prealloc > _size)
            prealloc = _size;

        for (int i = 0; i < prealloc; i++)
        {
            _pool.push_back(_allocator.alloc(obj));
        }
        pthread_mutex_unlock(&pool_lock);
    }
};
} // namespace generateTexGstbuf
#endif // MEMORYPOOL_H
